#ifndef __CControlPanel__
#define __CControlPanel__

#include "CControlRoot.hpp"
#include "CControl.hpp"

//	===========================================================================

using Exponent::GUI::Controls::CControlRoot;
using Exponent::GUI::Controls::CControl;

//	===========================================================================

namespace Exponent
{
	namespace GUI
	{
		namespace Controls
		{
			/**
			 * @class CControlPanel CControlPanel.hpp
			 * @brief A control panel is a control that holds other controls It inherits from both CControl and CControlRoot, allowing you to pass it controls to add as children
			 *
			 * @date 28/03/2005
			 * @author Paul Chana
			 * @version 1.0.0 Initial version
			 *
			 * @note All contents of this source code are copyright 2005 Exp Digital Uk.\n
			 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy\n
			 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
			 * All content is the Intellectual property of Exp Digital Uk.\n
			 * Certain sections of this code may come from other sources. They are credited where applicable.\n
			 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
			 *
			 * $Id: CControlPanel.hpp,v 1.7 2007/02/08 21:07:54 paul Exp $
			 */
			class CControlPanel : public CControlRoot, public CControl
			{
				/** @cond */
				EXPONENT_CLASS_DECLARATION;
				/** @endcond */

//	===========================================================================
		
			public:

//	===========================================================================

				/**
				 * Construction
				 * @param parent THe parent window holding this control root
				 * @param root The root control that this control is being added to
				 * @param uniqueId The unique Identifier of this control or CCONTROL_NO_ID_REQUIRED if no id is required
				 * @param area The area of this control relative to the root controls top left
				 * @param listener The action listener
				 * @note The root that is passed in to this constructor is the root for *this*, not for any sub children that are added to it.\n
				 * Under the Exponent API graphics system, every control must have a root control - This is the control that is responsible for delegating all\n
				 * mouse events, graphics calls etc. Generally for custom groupings of controls you will inherit from CControlPanel and then create custom controls\n
				 * inside it. To perform this action you would use this as the parent. For example:
				 *
				 * @code
				 * CMyControlPanel::CMyControlPanel(IWindow *parent, IControlRoot *root, const long uniqueId, const CRect &area, IActionListener *listener)
				 *                : CControlPanel(parent, root, uniqueId, area, listener)
				 * {
				 *		// Notice that we are being added to root, not our children....
				 *      
				 *      // Create our child controls, here we just create a generic control, but this could be any control that we choose
				 *		// Notice that the area is set to be relative to *our* top left..
				 *      CControl *control = new CControl(this, 10, CRect(0, 0, 10, 10), NULL);
				 *		// Now we add the control so that it will draw, recieve mouse moves etc...
				 *		this->addControl(control);
				 * }
				 * @endcode
				 */
				CControlPanel(IWindow *parent, IControlRoot *root, const long uniqueId, const CRect &area, IActionListener *listener = NULL);

				/**
				 * Destruction
				 */
				virtual ~CControlPanel();

//	===========================================================================

				/**
				 * Add a control
				 * @param control A control that becomes a child of this
				 */
				virtual void addControl(IControl *control);

				/**
				 * Lock a control (only this control recieves mouse info)
				 * @param control The control to feed all events to
				 */
				virtual void lockControl(IControl *control);

				/**
				 * Unlock a control
				 * @see lockControl
				 */
				virtual void unlockControl();
				
				/**
				 * Get window offset
				 * @retval CPoint The window offset for this panel
				 */
				virtual CPoint getWindowOffset();

				/**
				 * Is this control a container of other controls
				 * @retval bool True, this control can contain others
				 */
				virtual bool isContainer() const { return true; }

//	===========================================================================

				/**
				 * Get the global location of the control
				 * @param control The control to get the global coordinates of
				 * @param point On return contains the global position of the control
				 */
				virtual void getGlobalCoordinatesOfControl(IControl *control, CPoint &point);

				/**
				 * Set the size of the control
				 * @param area The are of the control
				 */
				virtual void setArea(const CRect &area);

//	===========================================================================

				/**
				 * Draw the root control
				 * @param graphics The graphics context
				 */
				virtual void drawRootControl(CGraphics &graphics);

//	===========================================================================

				/**
				 * Handle left button being clicked
				 * @param event The event to handle
				 */
				virtual void handleLeftButtonDown(CMouseEvent &event);

				/**
				 * Handle left button being released
				 * @param event The event to handle
				 */
				virtual void handleLeftButtonUp(CMouseEvent &event);

				/**
				 * Handle the right button being clicked
				 * @param event The event to handle
				 */
				virtual void handleRightButtonDown(CMouseEvent &event);

				/**
				 * Handle the right button being released
				 * @param event The event to handle
				 */
				virtual void handleRightButtonUp(CMouseEvent &event);

				/**
				 * Handle a double click on the left button
				 * @param event The event to handle
				 */
				virtual void handleDoubleClick(CMouseEvent &event);

				/**
				 * Handle the scroll wheel
				 * @param event The event to handle
				 */
				virtual void handleMouseScroll(CMouseEvent &event);

				/**
				 * Handle the mouse movement
				 * @param event The event to handle
				 */
				virtual void handleMouseMovement(CMouseEvent &event);

				/**
				 * Handle a mouse leaving movement
				 * @param event The event to handle
				 */
				virtual void handleMouseLeavingArea(CMouseEvent &event);

//	===========================================================================

				/**
				 * Handle key down events
				 * @param event The event to handle
				 */
				virtual bool handleKeyDown(const CKeyboardEvent &event);
				
				/**
				 * Handle key up events
				 * @param event The event to handle
				 */
				virtual bool handleKeyUp(const CKeyboardEvent &event);

//	===========================================================================

				/**
				 * Handle a file drop
				 * @param event The event to handle
				 */
				virtual void handleFileDrop(const CDropEvent &event);

//	===========================================================================

				/**
				 * Set the root control that this control is sitting on
				 * @param controlRoot The root control of this panel
				 */
				virtual void setControlRoot(IControlRoot *controlRoot);

				/**
				 * Draw the control
				 * @param graphics The graphics context
				 */
				virtual void drawControl(CGraphics &graphics);

				/**
				 * Update (redraw) a control
				 * @param control The control to redraw
				 */
				virtual void updateControl(IControl *control);

				/**
				 * Update an area
				 * @param area The area relative to our top left to redraw
				 */
				virtual void updateArea(const CRect &area);
				
				/**
				 * Enable a control
				 */
				virtual void enableControl() ;

				/**
				 * Disable the control
				 */
				virtual void disableControl();

				/**
				 * Should we draw the panel bounds?
				 * @param draw If true the panel bounding box will be drawn, if false will not. Useful during debugging
				 */
				virtual void onDrawDrawThePanelBounds(const bool draw = true) { m_drawPanelBounds = draw; }

				/**
				 * Should we clip children control
				 * @param clip If true clipping is applied, not recommended when placing inside a scroll panel
				 */
				void shouldClip(const bool clip = true);

				/**
				 * Set the colour of the bounds
				 * @param boundsColour The colour of the bounding box
				 */
				virtual void setBoundsColour(const CAlphaColour &boundsColour) { m_boundsColour = boundsColour; }

//	===========================================================================

				/**
				 * Reference
				 * @see ICountedObject
				 */
				virtual void referenced()
				{
					CControl::referenced();
				}

				/**
				 * Dereference
				 * @see ICountedObject
				 */
				virtual void dereference()
				{
					CControl::dereference();
				}

//	===========================================================================

				/**
				 * Get the window (absoloute) position of a control
				 * @param control The control to find the absolute position of
				 * @param point The position of the control on return
				 */
				virtual void getWindowCoordinatesOfControl(IControl *control, CPoint &point);

//	===========================================================================

			protected:

//	===========================================================================

				/**
				 * Check if the control is enabled. If it isnt, draw the disabled background and return
				 * @param graphics The graphics context
				 * @retval bool True if the control should be drawn, false otherwise
				 */
				virtual bool drawEnabledControl(CGraphics &graphics);

				/**
				 * Fluush throught a size change of some sort
				 */
				void flushSizeChange();

//	===========================================================================

				bool m_drawPanelBounds;					/**< Should we draw the bounds of the panel */
				CAlphaColour m_boundsColour;			/**< THe colour of the bounds */
			};
		}
	}
}
#endif	// End of CControlPanel.hpp